<?php
declare(strict_types=1);
require_once CLASSES_PATH . '/Database.php';

/**
 * ============================================================
 * FRESIL C.A. — Clase Vehiculo
 * Gestión de Marcas, Modelos y Años de vehículos (YMME)
 * ============================================================
 */
class Vehiculo
{
    private Database $db;

    public function __construct() { $this->db = Database::getInstance(); }

    // ── MARCAS ────────────────────────────────────────────────
    public function listarMarcas(bool $soloActivas = true): array
    {
        $where = $soloActivas ? 'WHERE activo = 1' : '';
        return $this->db->fetchAll(
            "SELECT id, nombre, pais_origen, activo FROM marcas_vehiculo {$where} ORDER BY nombre ASC"
        );
    }

    public function crearMarca(string $nombre, ?string $paisOrigen, int $userId): array
    {
        $nombre = trim(strip_tags($nombre));
        if (strlen($nombre) < 2 || strlen($nombre) > 80) {
            throw new RuntimeException('Nombre de marca debe tener entre 2 y 80 caracteres.', 422);
        }
        $existe = $this->db->fetchOne('SELECT id FROM marcas_vehiculo WHERE nombre=:n', [':n'=>$nombre]);
        if ($existe) { throw new RuntimeException("La marca '{$nombre}' ya existe.", 409); }

        $this->db->execute(
            'INSERT INTO marcas_vehiculo (nombre, pais_origen, activo) VALUES (:n, :p, 1)',
            [':n' => $nombre, ':p' => $paisOrigen]
        );
        $id = (int)$this->db->lastInsertId();
        $this->auditLog($userId, 'marcas_vehiculo', $id, 'INSERT');
        return $this->db->fetchOne('SELECT * FROM marcas_vehiculo WHERE id=:id', [':id'=>$id]);
    }

    public function actualizarMarca(int $id, string $nombre, ?string $paisOrigen, int $activo, int $userId): array
    {
        $nombre = trim(strip_tags($nombre));
        $existe = $this->db->fetchOne(
            'SELECT id FROM marcas_vehiculo WHERE nombre=:n AND id!=:id', [':n'=>$nombre,':id'=>$id]
        );
        if ($existe) { throw new RuntimeException("Nombre '{$nombre}' ya existe.", 409); }
        $this->db->execute(
            'UPDATE marcas_vehiculo SET nombre=:n, pais_origen=:p, activo=:a WHERE id=:id',
            [':n'=>$nombre, ':p'=>$paisOrigen, ':a'=>$activo, ':id'=>$id]
        );
        $this->auditLog($userId, 'marcas_vehiculo', $id, 'UPDATE');
        return $this->db->fetchOne('SELECT * FROM marcas_vehiculo WHERE id=:id',[':id'=>$id]);
    }

    // ── MODELOS ───────────────────────────────────────────────
    public function listarModelos(int $marcaId = 0, bool $soloActivos = true): array
    {
        $where  = $soloActivos ? ['activo = 1'] : [];
        $params = [];
        if ($marcaId > 0) {
            $where[]       = 'marca_id = :mid';
            $params[':mid'] = $marcaId;
        }
        $whereStr = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
        return $this->db->fetchAll(
            "SELECT m.*, mv.nombre AS marca_nombre
             FROM modelos_vehiculo m
             JOIN marcas_vehiculo mv ON mv.id = m.marca_id
             {$whereStr}
             ORDER BY mv.nombre, m.nombre ASC",
            $params
        );
    }

    public function crearModelo(int $marcaId, string $nombre, string $tipo, int $userId): array
    {
        $nombre = trim(strip_tags($nombre));
        if (strlen($nombre) < 2 || strlen($nombre) > 100) {
            throw new RuntimeException('Nombre de modelo debe tener entre 2 y 100 caracteres.', 422);
        }
        $tiposValidos = ['sedan','suv','pickup','van','coupe','hatchback','camioneta','bus','moto','otro'];
        if (!in_array($tipo, $tiposValidos, true)) { $tipo = 'sedan'; }

        $marca = $this->db->fetchOne('SELECT id FROM marcas_vehiculo WHERE id=:id',[':id'=>$marcaId]);
        if (!$marca) { throw new RuntimeException('Marca no encontrada.', 404); }

        $existe = $this->db->fetchOne(
            'SELECT id FROM modelos_vehiculo WHERE marca_id=:mid AND nombre=:n',
            [':mid'=>$marcaId,':n'=>$nombre]
        );
        if ($existe) { throw new RuntimeException("El modelo '{$nombre}' ya existe para esta marca.", 409); }

        $this->db->execute(
            'INSERT INTO modelos_vehiculo (marca_id,nombre,tipo,activo) VALUES (:mid,:n,:t,1)',
            [':mid'=>$marcaId,':n'=>$nombre,':t'=>$tipo]
        );
        $id = (int)$this->db->lastInsertId();
        $this->auditLog($userId,'modelos_vehiculo',$id,'INSERT');
        return $this->db->fetchOne('SELECT * FROM modelos_vehiculo WHERE id=:id',[':id'=>$id]);
    }

    // ── AÑOS / VERSIONES ──────────────────────────────────────
    public function listarAnios(int $modeloId): array
    {
        return $this->db->fetchAll(
            "SELECT va.*, mo.nombre AS modelo_nombre, mv.nombre AS marca_nombre
             FROM vehiculo_anios va
             JOIN modelos_vehiculo mo ON mo.id = va.modelo_id
             JOIN marcas_vehiculo  mv ON mv.id = mo.marca_id
             WHERE va.modelo_id = :mid
             ORDER BY va.anio_inicio ASC",
            [':mid' => $modeloId]
        );
    }

    public function crearAnio(int $modeloId, int $anioInicio, ?int $anioFin, ?string $motorCc, string $combustible, int $userId): array
    {
        $modelo = $this->db->fetchOne('SELECT id FROM modelos_vehiculo WHERE id=:id', [':id'=>$modeloId]);
        if (!$modelo) { throw new RuntimeException('Modelo no encontrado.', 404); }
        if ($anioInicio < 1950 || $anioInicio > (int)date('Y') + 2) {
            throw new RuntimeException('Año de inicio inválido.', 422);
        }
        if ($anioFin !== null && $anioFin < $anioInicio) {
            throw new RuntimeException('Año fin no puede ser menor al año inicio.', 422);
        }
        $combustibles = ['gasolina','diesel','hibrido','electrico'];
        if (!in_array($combustible, $combustibles, true)) { $combustible = 'gasolina'; }

        $this->db->execute(
            'INSERT INTO vehiculo_anios (modelo_id,anio_inicio,anio_fin,motor_cc,combustible)
             VALUES (:mid,:ai,:af,:mc,:comb)',
            [':mid'=>$modeloId,':ai'=>$anioInicio,':af'=>$anioFin,':mc'=>$motorCc,':comb'=>$combustible]
        );
        $id = (int)$this->db->lastInsertId();
        $this->auditLog($userId,'vehiculo_anios',$id,'INSERT');
        return $this->db->fetchOne(
            "SELECT va.*, mo.nombre AS modelo_nombre, mv.nombre AS marca_nombre
             FROM vehiculo_anios va
             JOIN modelos_vehiculo mo ON mo.id=va.modelo_id
             JOIN marcas_vehiculo  mv ON mv.id=mo.marca_id
             WHERE va.id=:id", [':id'=>$id]
        );
    }

    // ── BÚSQUEDA COMBINADA para selector YMME ─────────────────
    public function buscarVersiones(int $marcaId = 0, int $modeloId = 0, int $anio = 0): array
    {
        $where  = ['1=1'];
        $params = [];
        if ($marcaId  > 0) { $where[]='mo.marca_id=:mid';  $params[':mid']=$marcaId; }
        if ($modeloId > 0) { $where[]='va.modelo_id=:oid';  $params[':oid']=$modeloId; }
        if ($anio     > 0) {
            $where[]='(va.anio_inicio<=:ay AND (va.anio_fin IS NULL OR va.anio_fin>=:ay2))';
            $params[':ay']=$anio; $params[':ay2']=$anio;
        }
        return $this->db->fetchAll(
            "SELECT va.id, va.anio_inicio, va.anio_fin, va.motor_cc, va.combustible,
               mo.id AS modelo_id, mo.nombre AS modelo, mo.tipo,
               mv.id AS marca_id, mv.nombre AS marca
             FROM vehiculo_anios va
             JOIN modelos_vehiculo mo ON mo.id=va.modelo_id
             JOIN marcas_vehiculo  mv ON mv.id=mo.marca_id
             WHERE " . implode(' AND ', $where) . "
             ORDER BY mv.nombre, mo.nombre, va.anio_inicio",
            $params
        );
    }

    private function auditLog(int $userId, string $tabla, int $id, string $accion): void
    {
        try {
            $this->db->execute(
                'INSERT INTO auditoria_log (usuario_id,tabla_afectada,registro_id,accion,ip_address,created_at)
                 VALUES (:uid,:tabla,:id,:accion,:ip,NOW())',
                [':uid'=>$userId,':tabla'=>$tabla,':id'=>$id,':accion'=>$accion,':ip'=>$_SERVER['REMOTE_ADDR']??'']
            );
        } catch (\Exception) {}
    }
}
