<?php
declare(strict_types=1);

require_once CONFIG_PATH . '/config.php';

/**
 * ============================================================
 * FRESIL C.A. — Clase Database (PDO Singleton)
 * ─────────────────────────────────────────────────────────
 * • Conexión única reutilizable (patrón Singleton)
 * • 100% Prepared Statements — nunca concatenación SQL
 * • Charset utf8mb4 (soporta emojis y caracteres especiales)
 * • Modo excepción: cualquier error SQL lanza PDOException
 * ============================================================
 */
class Database
{
    private static ?Database $instance = null;
    private PDO $pdo;

    /**
     * Constructor privado — solo se llama desde getInstance()
     */
    private function __construct()
    {
        $dsn = sprintf(
            'mysql:host=%s;port=%s;dbname=%s;charset=%s',
            DB_HOST, DB_PORT, DB_NAME, DB_CHARSET
        );

        $options = [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,   // CRÍTICO: desactiva emulación → previene SQL injection
            PDO::ATTR_PERSISTENT         => false,
            PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci",
        ];

        try {
            $this->pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            // En producción no revelar detalles de la conexión
            $msg = (APP_ENV === 'development')
                ? 'Error de conexión BD: ' . $e->getMessage()
                : 'Error interno del servidor. Contacte al administrador.';
            throw new RuntimeException($msg, 500);
        }
    }

    /** Obtener la instancia única */
    public static function getInstance(): Database
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /** Obtener el objeto PDO directamente */
    public function getConnection(): PDO
    {
        return $this->pdo;
    }

    /**
     * Ejecutar un prepared statement de forma segura
     * @param string $sql   Query con placeholders (:param o ?)
     * @param array  $params Parámetros del query
     * @return PDOStatement
     */
    public function query(string $sql, array $params = []): PDOStatement
    {
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        return $stmt;
    }

    /**
     * Obtener todos los registros de una query
     */
    public function fetchAll(string $sql, array $params = []): array
    {
        return $this->query($sql, $params)->fetchAll();
    }

    /**
     * Obtener un solo registro
     */
    public function fetchOne(string $sql, array $params = []): array|false
    {
        return $this->query($sql, $params)->fetch();
    }

    /**
     * Ejecutar INSERT/UPDATE/DELETE y obtener filas afectadas
     */
    public function execute(string $sql, array $params = []): int
    {
        return $this->query($sql, $params)->rowCount();
    }

    /**
     * Obtener el último ID insertado
     */
    public function lastInsertId(): string
    {
        return $this->pdo->lastInsertId();
    }

    /**
     * Iniciar transacción
     */
    public function beginTransaction(): void
    {
        $this->pdo->beginTransaction();
    }

    /**
     * Confirmar transacción
     */
    public function commit(): void
    {
        $this->pdo->commit();
    }

    /**
     * Revertir transacción
     */
    public function rollback(): void
    {
        $this->pdo->rollBack();
    }

    // Prevenir clonación y deserialización
    private function __clone() {}
    public function __wakeup(): void
    {
        throw new \Exception('No se puede deserializar el Singleton Database.');
    }
}
