<?php
declare(strict_types=1);
require_once CLASSES_PATH . '/Database.php';

/**
 * FRESIL C.A. — Clase Categoria
 * Árbol jerárquico de categorías de repuestos
 */
class Categoria
{
    private Database $db;

    public function __construct() { $this->db = Database::getInstance(); }

    /** Listar todas las categorías como árbol */
    public function arbol(bool $soloActivas = true): array
    {
        $where = $soloActivas ? 'WHERE activo = 1' : '';
        $all = $this->db->fetchAll(
            "SELECT id, parent_id, nombre, descripcion, icono, activo, orden
             FROM categorias_repuesto {$where}
             ORDER BY parent_id IS NULL DESC, orden ASC, nombre ASC"
        );

        // Construir árbol en PHP
        $mapa    = [];
        $raices  = [];
        foreach ($all as $c) {
            $c['hijos'] = [];
            $mapa[$c['id']] = $c;
        }
        foreach ($mapa as &$c) {
            if ($c['parent_id'] === null) {
                $raices[] = &$c;
            } else {
                if (isset($mapa[$c['parent_id']])) {
                    $mapa[$c['parent_id']]['hijos'][] = &$c;
                }
            }
        }
        return $raices;
    }

    /** Listar plana (para selects) */
    public function listarPlana(bool $soloActivas = true): array
    {
        $where = $soloActivas ? 'WHERE activo=1' : '';
        return $this->db->fetchAll(
            "SELECT id, parent_id, nombre, icono, activo
             FROM categorias_repuesto {$where}
             ORDER BY parent_id IS NULL DESC, orden ASC, nombre ASC"
        );
    }

    public function obtener(int $id): array
    {
        $c = $this->db->fetchOne('SELECT * FROM categorias_repuesto WHERE id=:id',[':id'=>$id]);
        if (!$c) { throw new RuntimeException("Categoría #{$id} no encontrada.", 404); }
        return $c;
    }

    public function crear(array $d, int $userId): array
    {
        $nombre = trim(strip_tags((string)($d['nombre']??'')));
        if (strlen($nombre) < 2) { throw new RuntimeException('Nombre muy corto.', 422); }
        $parentId = isset($d['parent_id']) && $d['parent_id'] ? (int)$d['parent_id'] : null;
        $icono    = trim((string)($d['icono']??''));

        $this->db->execute(
            'INSERT INTO categorias_repuesto (parent_id,nombre,descripcion,icono,activo,orden)
             VALUES (:pid,:n,:desc,:ico,1,:ord)',
            [':pid'=>$parentId,':n'=>$nombre,':desc'=>$d['descripcion']??null,
             ':ico'=>$icono?:null,':ord'=>(int)($d['orden']??0)]
        );
        $id = (int)$this->db->lastInsertId();
        $this->auditLog($userId,$id,'INSERT');
        return $this->obtener($id);
    }

    public function actualizar(int $id, array $d, int $userId): array
    {
        $this->obtener($id);
        $nombre = trim(strip_tags((string)($d['nombre']??'')));
        if (strlen($nombre) < 2) { throw new RuntimeException('Nombre muy corto.', 422); }
        $this->db->execute(
            'UPDATE categorias_repuesto SET nombre=:n,descripcion=:desc,icono=:ico,activo=:a,orden=:ord WHERE id=:id',
            [':n'=>$nombre,':desc'=>$d['descripcion']??null,':ico'=>$d['icono']??null,
             ':a'=>(int)(bool)($d['activo']??1),':ord'=>(int)($d['orden']??0),':id'=>$id]
        );
        $this->auditLog($userId,$id,'UPDATE');
        return $this->obtener($id);
    }

    private function auditLog(int $userId, int $id, string $accion): void
    {
        try {
            $this->db->execute(
                'INSERT INTO auditoria_log (usuario_id,tabla_afectada,registro_id,accion,ip_address,created_at)
                 VALUES (:uid,"categorias_repuesto",:id,:accion,:ip,NOW())',
                [':uid'=>$userId,':id'=>$id,':accion'=>$accion,':ip'=>$_SERVER['REMOTE_ADDR']??'']
            );
        } catch(\Exception) {}
    }
}
