<?php
declare(strict_types=1);

require_once CLASSES_PATH . '/Database.php';
require_once CLASSES_PATH . '/JWT.php';

/**
 * ============================================================
 * FRESIL C.A. — Clase Auth
 * ─────────────────────────────────────────────────────────
 * Maneja toda la lógica de autenticación:
 * • Login con rate limiting y bloqueo de cuenta
 * • Generación y renovación de tokens JWT
 * • Refresh tokens almacenados en BD (hash SHA-256)
 * • Logout con invalidación de token
 * • Registro en auditoría de cada evento
 * ============================================================
 */
class Auth
{
    private Database $db;

    public function __construct()
    {
        $this->db = Database::getInstance();
    }

    /**
     * ── Autenticar usuario ────────────────────────────────────
     *
     * @param string $email
     * @param string $password
     * @param string $ipAddress  IP del cliente
     * @return array ['access_token', 'refresh_token', 'usuario']
     * @throws RuntimeException
     */
    public function login(string $email, string $password, string $ipAddress): array
    {
        // 1. Sanitizar inputs
        $email = strtolower(trim(filter_var($email, FILTER_SANITIZE_EMAIL)));

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new RuntimeException('Credenciales inválidas.', 401);
        }

        // 2. Verificar si la IP está bloqueada por rate limiting
        $this->checkRateLimit($ipAddress, $email);

        // 3. Buscar usuario en BD
        $usuario = $this->db->fetchOne(
            'SELECT id, nombre, email, password_hash, rol, activo
             FROM usuarios
             WHERE email = :email
             LIMIT 1',
            [':email' => $email]
        );

        // 4. Si no existe o contraseña incorrecta → mismo error genérico (prevenir enumeración)
        if (!$usuario || !password_verify($password, $usuario['password_hash'])) {
            $this->recordFailedAttempt($ipAddress, $email);
            throw new RuntimeException('Credenciales inválidas.', 401);
        }

        // 5. Verificar que la cuenta esté activa
        if (!(bool)$usuario['activo']) {
            throw new RuntimeException('Cuenta desactivada. Contacte al administrador.', 403);
        }

        // 6. Limpiar intentos fallidos (login exitoso)
        $this->clearFailedAttempts($ipAddress, $email);

        // 7. Verificar si el hash necesita actualización (upgrading bcrypt cost)
        if (password_needs_rehash($usuario['password_hash'], PASSWORD_BCRYPT, ['cost' => 12])) {
            $newHash = password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
            $this->db->execute(
                'UPDATE usuarios SET password_hash = :hash WHERE id = :id',
                [':hash' => $newHash, ':id' => $usuario['id']]
            );
        }

        // 8. Actualizar último login
        $this->db->execute(
            'UPDATE usuarios SET ultimo_login = NOW() WHERE id = :id',
            [':id' => $usuario['id']]
        );

        // 9. Generar tokens
        $payload = [
            'sub'    => (int)$usuario['id'],
            'email'  => $usuario['email'],
            'nombre' => $usuario['nombre'],
            'rol'    => $usuario['rol'],
        ];

        $accessToken  = JWT::generate($payload);
        $refreshToken = JWT::generateRefreshToken();

        // 10. Almacenar refresh token en BD (solo el hash)
        $this->saveRefreshToken((int)$usuario['id'], $refreshToken, $ipAddress);

        // 11. Registrar en auditoría
        $this->auditLog(
            userId: (int)$usuario['id'],
            tabla:  'usuarios',
            regId:  (int)$usuario['id'],
            accion: 'LOGIN',
            ip:     $ipAddress,
            after:  ['evento' => 'login_exitoso']
        );

        // 12. Retornar — NO incluir password_hash en la respuesta
        unset($usuario['password_hash']);

        return [
            'access_token'  => $accessToken,
            'refresh_token' => $refreshToken,
            'expires_in'    => JWT_EXPIRE_MINUTES * 60,
            'token_type'    => 'Bearer',
            'usuario'       => $usuario,
        ];
    }

    /**
     * ── Renovar access token con refresh token ───────────────
     */
    public function refresh(string $refreshToken, string $ipAddress): array
    {
        if (empty($refreshToken)) {
            throw new RuntimeException('Refresh token requerido.', 401);
        }

        $tokenHash = hash('sha256', $refreshToken);

        $stored = $this->db->fetchOne(
            'SELECT rt.*, u.id as uid, u.nombre, u.email, u.rol, u.activo
             FROM refresh_tokens rt
             JOIN usuarios u ON u.id = rt.usuario_id
             WHERE rt.token_hash = :hash
               AND rt.revocado = 0
               AND rt.expira_en > NOW()
             LIMIT 1',
            [':hash' => $tokenHash]
        );

        if (!$stored) {
            throw new RuntimeException('Refresh token inválido o expirado.', 401);
        }

        if (!(bool)$stored['activo']) {
            throw new RuntimeException('Cuenta desactivada.', 403);
        }

        // Revocar el refresh token actual (rotación de tokens)
        $this->db->execute(
            'UPDATE refresh_tokens SET revocado = 1, revocado_en = NOW() WHERE token_hash = :hash',
            [':hash' => $tokenHash]
        );

        // Generar nuevos tokens
        $payload = [
            'sub'    => (int)$stored['uid'],
            'email'  => $stored['email'],
            'nombre' => $stored['nombre'],
            'rol'    => $stored['rol'],
        ];

        $newAccessToken  = JWT::generate($payload);
        $newRefreshToken = JWT::generateRefreshToken();

        $this->saveRefreshToken((int)$stored['uid'], $newRefreshToken, $ipAddress);

        return [
            'access_token'  => $newAccessToken,
            'refresh_token' => $newRefreshToken,
            'expires_in'    => JWT_EXPIRE_MINUTES * 60,
            'token_type'    => 'Bearer',
        ];
    }

    /**
     * ── Logout: revocar refresh token ────────────────────────
     */
    public function logout(string $refreshToken, int $userId, string $ipAddress): void
    {
        if (!empty($refreshToken)) {
            $tokenHash = hash('sha256', $refreshToken);
            $this->db->execute(
                'UPDATE refresh_tokens
                 SET revocado = 1, revocado_en = NOW()
                 WHERE token_hash = :hash AND usuario_id = :uid',
                [':hash' => $tokenHash, ':uid' => $userId]
            );
        }

        $this->auditLog($userId, 'usuarios', $userId, 'LOGOUT', $ipAddress);
    }

    /**
     * ── Obtener usuario autenticado desde JWT ─────────────────
     */
    public function getCurrentUser(string $bearerToken): array
    {
        $token   = $this->extractBearerToken($bearerToken);
        $payload = JWT::validate($token);

        $usuario = $this->db->fetchOne(
            'SELECT id, nombre, email, rol, activo, ultimo_login, created_at
             FROM usuarios
             WHERE id = :id AND activo = 1
             LIMIT 1',
            [':id' => (int)$payload['sub']]
        );

        if (!$usuario) {
            throw new RuntimeException('Usuario no encontrado o inactivo.', 401);
        }

        return $usuario;
    }

    /**
     * ── Extraer token Bearer del header Authorization ─────────
     */
    public function extractBearerToken(string $header): string
    {
        if (preg_match('/^Bearer\s+(.+)$/i', trim($header), $matches)) {
            return $matches[1];
        }
        throw new RuntimeException('Token de autorización no encontrado.', 401);
    }

    // ─────────────────────────────────────────────────────────
    // Métodos privados internos
    // ─────────────────────────────────────────────────────────

    /** Almacenar refresh token (hash SHA-256, nunca texto plano) */
    private function saveRefreshToken(int $userId, string $token, string $ipAddress): void
    {
        $tokenHash = hash('sha256', $token);
        $expiresAt = date('Y-m-d H:i:s', strtotime('+' . JWT_REFRESH_DAYS . ' days'));

        $this->db->execute(
            'INSERT INTO refresh_tokens
                (usuario_id, token_hash, ip_address, expira_en, creado_en)
             VALUES
                (:uid, :hash, :ip, :exp, NOW())',
            [
                ':uid'  => $userId,
                ':hash' => $tokenHash,
                ':ip'   => $ipAddress,
                ':exp'  => $expiresAt,
            ]
        );
    }

    /** Verificar rate limiting — bloquear si supera límite */
    private function checkRateLimit(string $ipAddress, string $email): void
    {
        $lockoutTime = date('Y-m-d H:i:s', strtotime('-' . LOGIN_LOCKOUT_MINS . ' minutes'));

        $intentos = $this->db->fetchOne(
            'SELECT COUNT(*) as total
             FROM login_intentos
             WHERE (ip_address = :ip OR email = :email)
               AND exitoso = 0
               AND created_at > :lockout',
            [':ip' => $ipAddress, ':email' => $email, ':lockout' => $lockoutTime]
        );

        if ((int)$intentos['total'] >= LOGIN_MAX_ATTEMPTS) {
            $minutos = LOGIN_LOCKOUT_MINS;
            throw new RuntimeException(
                "Cuenta bloqueada temporalmente. Intente en {$minutos} minutos.",
                429
            );
        }
    }

    /** Registrar intento fallido */
    private function recordFailedAttempt(string $ipAddress, string $email): void
    {
        $this->db->execute(
            'INSERT INTO login_intentos (ip_address, email, exitoso, created_at)
             VALUES (:ip, :email, 0, NOW())',
            [':ip' => $ipAddress, ':email' => $email]
        );
    }

    /** Limpiar intentos fallidos tras login exitoso */
    private function clearFailedAttempts(string $ipAddress, string $email): void
    {
        $this->db->execute(
            'DELETE FROM login_intentos
             WHERE ip_address = :ip OR email = :email',
            [':ip' => $ipAddress, ':email' => $email]
        );
    }

    /** Escribir en tabla de auditoría */
    private function auditLog(
        int    $userId,
        string $tabla,
        int    $regId,
        string $accion,
        string $ip,
        array  $before = [],
        array  $after  = []
    ): void {
        try {
            $this->db->execute(
                'INSERT INTO auditoria_log
                    (usuario_id, tabla_afectada, registro_id, accion, datos_antes, datos_despues, ip_address, user_agent, created_at)
                 VALUES
                    (:uid, :tabla, :regid, :accion, :antes, :despues, :ip, :ua, NOW())',
                [
                    ':uid'     => $userId,
                    ':tabla'   => $tabla,
                    ':regid'   => $regId,
                    ':accion'  => $accion,
                    ':antes'   => !empty($before)  ? json_encode($before,  JSON_UNESCAPED_UNICODE) : null,
                    ':despues' => !empty($after)   ? json_encode($after,   JSON_UNESCAPED_UNICODE) : null,
                    ':ip'      => $ip,
                    ':ua'      => $_SERVER['HTTP_USER_AGENT'] ?? null,
                ]
            );
        } catch (\Exception) {
            // La auditoría no debe detener el flujo principal
        }
    }
}
