<?php
declare(strict_types=1);
/**
 * FRESIL C.A. — API /api/vehiculos/index.php
 *
 * GET ?recurso=marcas             → listar marcas
 * GET ?recurso=modelos&marca_id=N → modelos de una marca
 * GET ?recurso=anios&modelo_id=N  → años/versiones de un modelo
 * GET ?recurso=versiones&marca_id=N&modelo_id=N&anio=N → búsqueda YMME
 *
 * POST ?recurso=marcas            → crear marca  (admin)
 * POST ?recurso=modelos           → crear modelo (admin)
 * POST ?recurso=anios             → crear año    (admin)
 */
require_once dirname(__DIR__,2).'/config/config.php';
require_once CLASSES_PATH.'/Database.php';
require_once CLASSES_PATH.'/JWT.php';
require_once CLASSES_PATH.'/Vehiculo.php';
require_once ROOT_PATH.'/middleware/AuthMiddleware.php';

header('Content-Type: application/json; charset=utf-8');
if ($_SERVER['REQUEST_METHOD']==='OPTIONS'){http_response_code(204);exit;}

$method   = $_SERVER['REQUEST_METHOD'];
$recurso  = trim((string)($_GET['recurso'] ?? ''));

try {
    $user  = AuthMiddleware::requireAuth();
    $veh   = new Vehiculo();

    if ($method === 'GET') {
        switch ($recurso) {
            case 'marcas':
                AuthMiddleware::respond($veh->listarMarcas());
                break;
            case 'modelos':
                $marcaId = (int)($_GET['marca_id'] ?? 0);
                AuthMiddleware::respond($veh->listarModelos($marcaId));
                break;
            case 'anios':
                $modeloId = (int)($_GET['modelo_id'] ?? 0);
                if (!$modeloId) { AuthMiddleware::abort(400,'modelo_id requerido.'); }
                AuthMiddleware::respond($veh->listarAnios($modeloId));
                break;
            case 'versiones':
                AuthMiddleware::respond($veh->buscarVersiones(
                    marcaId:  (int)($_GET['marca_id']  ?? 0),
                    modeloId: (int)($_GET['modelo_id'] ?? 0),
                    anio:     (int)($_GET['anio']      ?? 0),
                ));
                break;
            default:
                AuthMiddleware::abort(400, 'Recurso inválido. Use: marcas, modelos, anios, versiones');
        }
    } elseif ($method === 'POST') {
        AuthMiddleware::requireRole(['admin','almacen']);
        $body = json_decode(file_get_contents('php://input'), true, 512, JSON_THROW_ON_ERROR);
        if (!is_array($body)) { AuthMiddleware::abort(400,'JSON inválido.'); }

        switch ($recurso) {
            case 'marcas':
                AuthMiddleware::requireRole('admin');
                $resultado = $veh->crearMarca(
                    $body['nombre']      ?? '',
                    $body['pais_origen'] ?? null,
                    (int)$user['id']
                );
                AuthMiddleware::respond($resultado, 201);
                break;
            case 'modelos':
                $resultado = $veh->crearModelo(
                    (int)($body['marca_id'] ?? 0),
                    $body['nombre'] ?? '',
                    $body['tipo']   ?? 'sedan',
                    (int)$user['id']
                );
                AuthMiddleware::respond($resultado, 201);
                break;
            case 'anios':
                $resultado = $veh->crearAnio(
                    (int)($body['modelo_id']   ?? 0),
                    (int)($body['anio_inicio'] ?? 0),
                    !empty($body['anio_fin']) ? (int)$body['anio_fin'] : null,
                    $body['motor_cc']    ?? null,
                    $body['combustible'] ?? 'gasolina',
                    (int)$user['id']
                );
                AuthMiddleware::respond($resultado, 201);
                break;
            default:
                AuthMiddleware::abort(400,'Recurso POST inválido.');
        }
    } else {
        AuthMiddleware::abort(405,'Método no permitido.');
    }
} catch (RuntimeException $e) {
    $c = in_array($e->getCode(),[400,401,403,404,409,422]) ? $e->getCode() : 500;
    AuthMiddleware::abort($c,$e->getMessage());
} catch(\JsonException) { AuthMiddleware::abort(400,'JSON inválido.'); }
  catch(\Throwable $e) {
    AuthMiddleware::abort(500, APP_ENV==='development'?$e->getMessage():'Error interno.');
}
