<?php
declare(strict_types=1);

/**
 * ============================================================
 * FRESIL C.A. — API: /api/usuarios/index.php
 * ─────────────────────────────────────────────────────────
 * GET    /api/usuarios/          → Listar (admin)
 * POST   /api/usuarios/          → Crear  (admin)
 * GET    /api/usuarios/?id=N     → Obtener uno (admin)
 * PUT    /api/usuarios/?id=N     → Actualizar (admin)
 * DELETE /api/usuarios/?id=N     → Desactivar (admin)
 *
 * POST   /api/usuarios/password.php → Cambiar propia contraseña
 * ============================================================
 */

require_once dirname(__DIR__, 2) . '/config/config.php';
require_once CLASSES_PATH . '/Database.php';
require_once CLASSES_PATH . '/JWT.php';
require_once CLASSES_PATH . '/Auth.php';
require_once CLASSES_PATH . '/Usuario.php';
require_once ROOT_PATH . '/middleware/AuthMiddleware.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: ' . APP_URL);
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { http_response_code(204); exit; }

$ip     = AuthMiddleware::getClientIp();
$method = $_SERVER['REQUEST_METHOD'];
$id     = isset($_GET['id']) ? (int)$_GET['id'] : null;

try {
    // Todos los endpoints de usuarios requieren rol admin
    $admin = AuthMiddleware::requireRole('admin');

    $modelo = new Usuario();

    switch ($method) {

        // ── GET: Listar o Obtener uno ─────────────────────────
        case 'GET':
            if ($id !== null) {
                // Obtener usuario específico
                AuthMiddleware::respond($modelo->obtener($id));
            } else {
                // Listar con paginación
                $pagina   = max(1, (int)($_GET['pagina']    ?? 1));
                $porPag   = min(100, (int)($_GET['por_pagina'] ?? 20));
                $buscar   = trim(strip_tags((string)($_GET['buscar'] ?? '')));
                $rol      = trim((string)($_GET['rol']    ?? ''));
                $activo   = isset($_GET['activo']) ? (int)$_GET['activo'] : -1;

                AuthMiddleware::respond($modelo->listar($pagina, $porPag, $buscar, $rol, $activo));
            }
            break;

        // ── POST: Crear usuario ───────────────────────────────
        case 'POST':
            $body = json_decode(file_get_contents('php://input'), true, flags: JSON_THROW_ON_ERROR);
            if (!is_array($body)) { AuthMiddleware::abort(400, 'JSON inválido.'); }

            $nuevo = $modelo->crear($body, (int)$admin['id'], $ip);
            AuthMiddleware::respond($nuevo, 201);
            break;

        // ── PUT: Actualizar usuario ───────────────────────────
        case 'PUT':
            if (!$id) { AuthMiddleware::abort(400, 'ID de usuario requerido.'); }
            $body = json_decode(file_get_contents('php://input'), true, flags: JSON_THROW_ON_ERROR);
            if (!is_array($body)) { AuthMiddleware::abort(400, 'JSON inválido.'); }

            $actualizado = $modelo->actualizar($id, $body, (int)$admin['id'], $ip);
            AuthMiddleware::respond($actualizado);
            break;

        // ── DELETE: Desactivar usuario ────────────────────────
        case 'DELETE':
            if (!$id) { AuthMiddleware::abort(400, 'ID de usuario requerido.'); }
            $modelo->desactivar($id, (int)$admin['id'], $ip);
            AuthMiddleware::respond(['mensaje' => "Usuario #{$id} desactivado."]);
            break;

        default:
            AuthMiddleware::abort(405, 'Método no permitido.');
    }

} catch (RuntimeException $e) {
    $code = in_array($e->getCode(), [400, 401, 403, 404, 409, 422, 429]) ? $e->getCode() : 500;
    AuthMiddleware::abort($code, $e->getMessage());
} catch (\JsonException) {
    AuthMiddleware::abort(400, 'JSON inválido en el cuerpo de la petición.');
} catch (\Throwable $e) {
    $msg = (APP_ENV === 'development') ? $e->getMessage() : 'Error interno del servidor.';
    AuthMiddleware::abort(500, $msg);
}
