<?php
declare(strict_types=1);
/**
 * FRESIL C.A. — API /api/repuestos/index.php
 * GET    → listar (con filtros)
 * POST   → crear
 * GET?id=N → obtener uno
 * PUT?id=N → actualizar
 * DELETE?id=N → eliminar
 */
require_once dirname(__DIR__,2).'/config/config.php';
require_once CLASSES_PATH.'/Database.php';
require_once CLASSES_PATH.'/JWT.php';
require_once CLASSES_PATH.'/Repuesto.php';
require_once ROOT_PATH.'/middleware/AuthMiddleware.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: '.APP_URL);
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if ($_SERVER['REQUEST_METHOD']==='OPTIONS'){http_response_code(204);exit;}

$method = $_SERVER['REQUEST_METHOD'];
$id     = isset($_GET['id']) ? (int)$_GET['id'] : null;

try {
    $user   = AuthMiddleware::requireAuth();
    $modelo = new Repuesto();

    switch ($method) {
        case 'GET':
            // Estadísticas
            if (isset($_GET['stats'])) {
                AuthMiddleware::respond($modelo->estadisticas());
            }
            // Búsqueda por barcode
            if (isset($_GET['barcode'])) {
                $cod = trim((string)($_GET['barcode']??''));
                AuthMiddleware::respond($modelo->buscarPorBarcode($cod));
            }
            // Generar siguiente código
            if (isset($_GET['siguiente_codigo'])) {
                AuthMiddleware::respond(['codigo' => $modelo->generarCodigo()]);
            }
            // Obtener uno
            if ($id !== null) {
                AuthMiddleware::respond($modelo->obtener($id));
            }
            // Listar con filtros
            AuthMiddleware::respond($modelo->listar(
                pagina:      max(1,(int)($_GET['pagina']    ??1)),
                porPagina:   min(100,(int)($_GET['por_pagina']??24)),
                buscar:      trim(strip_tags((string)($_GET['buscar']    ??' '))),
                categoriaId: (int)($_GET['categoria_id']??0),
                marcaId:     (int)($_GET['marca_id']     ??0),
                modeloId:    (int)($_GET['modelo_id']    ??0),
                anio:        (int)($_GET['anio']         ??0),
                soloStockBajo:(int)($_GET['stock_bajo']  ??0),
                activo:      isset($_GET['activo']) ? (int)$_GET['activo'] : 1,
                orden:       trim((string)($_GET['orden']?? 'created_at')),
                dir:         trim((string)($_GET['dir']  ?? 'DESC')),
            ));
            break;

        case 'POST':
            // Solo admin y almacen pueden crear
            if (!in_array($user['rol'],['admin','almacen'],true)) {
                AuthMiddleware::abort(403,'No tiene permisos para crear repuestos.');
            }
            $body = json_decode(file_get_contents('php://input'), true, 512, JSON_THROW_ON_ERROR);
            if (!is_array($body)) { AuthMiddleware::abort(400,'JSON inválido.'); }
            AuthMiddleware::respond($modelo->crear($body,(int)$user['id']), 201);
            break;

        case 'PUT':
            if (!$id) { AuthMiddleware::abort(400,'ID requerido.'); }
            if (!in_array($user['rol'],['admin','almacen'],true)) {
                AuthMiddleware::abort(403,'No tiene permisos para modificar repuestos.');
            }
            $body = json_decode(file_get_contents('php://input'), true, 512, JSON_THROW_ON_ERROR);
            if (!is_array($body)) { AuthMiddleware::abort(400,'JSON inválido.'); }
            AuthMiddleware::respond($modelo->actualizar($id,$body,(int)$user['id']));
            break;

        case 'DELETE':
            if (!$id) { AuthMiddleware::abort(400,'ID requerido.'); }
            AuthMiddleware::requireRole(['admin','almacen']);
            $modelo->eliminar($id,(int)$user['id']);
            AuthMiddleware::respond(['mensaje'=>"Repuesto #{$id} eliminado."]);
            break;

        default: AuthMiddleware::abort(405,'Método no permitido.');
    }
} catch (RuntimeException $e) {
    $c = in_array($e->getCode(),[400,401,403,404,409,422,429]) ? $e->getCode() : 500;
    AuthMiddleware::abort($c,$e->getMessage());
} catch (\JsonException) { AuthMiddleware::abort(400,'JSON inválido.'); }
  catch (\Throwable $e) {
    $msg = APP_ENV==='development' ? $e->getMessage() : 'Error interno del servidor.';
    AuthMiddleware::abort(500,$msg);
}
