<?php
declare(strict_types=1);

/**
 * ============================================================
 * FRESIL C.A. — API: POST /api/auth/login.php
 * ─────────────────────────────────────────────────────────
 * Retorna: access_token (JWT), refresh_token, datos del usuario
 * ============================================================
 */

require_once dirname(__DIR__, 2) . '/config/config.php';
require_once CLASSES_PATH . '/Database.php';
require_once CLASSES_PATH . '/JWT.php';
require_once CLASSES_PATH . '/Auth.php';
require_once ROOT_PATH . '/middleware/AuthMiddleware.php';

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: ' . APP_URL);
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Preflight CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

// Solo aceptar POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    AuthMiddleware::abort(405, 'Método no permitido.');
}

try {
    // Leer body JSON
    $body = file_get_contents('php://input');
    $input = json_decode($body, associative: true, flags: JSON_THROW_ON_ERROR);

    if (!is_array($input)) {
        AuthMiddleware::abort(400, 'Cuerpo de la petición inválido (se esperaba JSON).');
    }

    // Validar campos obligatorios presentes
    $email    = trim((string)($input['email']    ?? ''));
    $password = (string)($input['password'] ?? '');

    if (empty($email) || empty($password)) {
        AuthMiddleware::abort(400, 'Email y contraseña son requeridos.');
    }

    $ip   = AuthMiddleware::getClientIp();
    $auth = new Auth();

    $resultado = $auth->login($email, $password, $ip);

    AuthMiddleware::respond($resultado, 200);

} catch (RuntimeException $e) {
    $code = in_array($e->getCode(), [400, 401, 403, 404, 409, 422, 429]) ? $e->getCode() : 500;
    AuthMiddleware::abort($code, $e->getMessage());
} catch (\JsonException $e) {
    AuthMiddleware::abort(400, 'JSON inválido en el cuerpo de la petición.');
} catch (\Throwable $e) {
    $msg = (APP_ENV === 'development') ? $e->getMessage() : 'Error interno del servidor.';
    AuthMiddleware::abort(500, $msg);
}
